import os, re, sys, json, urllib.request

ORG   = os.environ.get("HF_ORG", "Trustcat")
TOKEN = os.environ.get("HUGGINGFACE_TOKEN", "")

def fetch_count(kind: str) -> int:
    # kind in {"models","datasets","spaces"}
    total, start, limit = 0, 0, 100
    headers = {"Accept":"application/json"}
    if TOKEN: headers["Authorization"] = f"Bearer {TOKEN}"
    while True:
        url = f"https://huggingface.co/api/{kind}?author={ORG}&limit={limit}&full=false&start={start}"
        req = urllib.request.Request(url, headers=headers)
        with urllib.request.urlopen(req) as resp:
            items = json.load(resp)
        n = len(items)
        total += n
        if n < limit: break
        start += limit
    return total

def build_badges(models, datasets, spaces):
    # link targets
    org_url = f"https://huggingface.co/{ORG}"
    return (
        '<p align="left">\n'
        f'  <a href="{org_url}" target="_blank" rel="noopener">\n'
        f'    <img alt="HF Models"   src="https://img.shields.io/badge/Models-{models}-FFD21E?logo=huggingface&logoColor=black">\n'
        f'  </a>\n'
        f'  <a href="{org_url}" target="_blank" rel="noopener">\n'
        f'    <img alt="HF Datasets" src="https://img.shields.io/badge/Datasets-{datasets}-0ea5e9?logo=huggingface&logoColor=black">\n'
        f'  </a>\n'
        f'  <a href="{org_url}" target="_blank" rel="noopener">\n'
        f'    <img alt="HF Spaces"   src="https://img.shields.io/badge/Spaces-{spaces}-a78bfa?logo=huggingface&logoColor=black">\n'
        f'  </a>\n'
        '</p>\n'
    )

START = "<!--HF:START-->"
END   = "<!--HF:END-->"

# counts (handle HF rate limits gracefully)
try:
    m = fetch_count("models")
    d = fetch_count("datasets")
    s = fetch_count("spaces")
except Exception as e:
    print("WARN: HF API error:", e, file=sys.stderr)
    # keep previously rendered block if present; otherwise show zeros
    m = d = s = 0

row = build_badges(m, d, s)
block = f"{START}\n{row}{END}"

path = "README.md"
try:
    with open(path, "r", encoding="utf-8") as f:
        readme = f.read()
except FileNotFoundError:
    readme = "# TrustCat\n\n"

if START in readme and END in readme:
    new = re.sub(re.escape(START)+r".*?"+re.escape(END), block, readme, flags=re.S)
else:
    new = readme.rstrip() + "\n\n## Models (Hugging Face)\n" + block + "\n"

if new != readme:
    with open(path, "w", encoding="utf-8") as f:
        f.write(new)
    print("README updated with HF badges.")
else:
    print("No README change.")
